/** @file   fpstimer.cpp
 * @brief   Implementation of FpsTimer - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "fpstimer.h"
#include <allegro.h>

namespace eng2d {

///
/// Static members, constants and datatypes
/// =======================================

const int FpsTimer::KNone;
const int FpsTimer::KBpsOutOfRange;
const int FpsTimer::KAllegroError;

bool FpsTimer::timerInstalled = false;
volatile int FpsTimer::updateCounter = 0;
volatile unsigned int FpsTimer::numOfUpdates = 0;
volatile unsigned int FpsTimer::logicalUpdatesCountVar = 0;

FpsTimer* FpsTimer::s_timerInstance = 0;
int FpsTimer::numberOfInstances = 0;



/** Timer interrupt handler.
 */
void FpsTimer::timerUpdater()
{
  updateCounter += 1;
  numOfUpdates += 1;
}
END_OF_STATIC_FUNCTION( FpsTimer::timerUpdater );



/** Installs the timer
 */
int FpsTimer::install( int aBps )
{
  if ( timerInstalled == true ) {
    return KNone;
  }
  
  if ( aBps < 1 || aBps > 1000 ) {
    return KBpsOutOfRange;
  }
  
  LOCK_VARIABLE( FpsTimer::updateCounter );
  LOCK_VARIABLE( FpsTimer::numOfUpdates );
  LOCK_VARIABLE( FpsTimer::logicalUpdatesCountVar );
  LOCK_FUNCTION( FpsTimer::timerUpdater );
  if ( install_int_ex( FpsTimer::timerUpdater, BPS_TO_TIMER( aBps ) ) < 0 ) {
    return KAllegroError;
  }
  timerInstalled = true;
  return KNone;
}

void FpsTimer::remove()
{
  if ( timerInstalled == false ) {
    return;
  }
  remove_int( FpsTimer::timerUpdater );
  timerInstalled = false;
}



/** Returns the pointer to timer object.
 */
FpsTimer* FpsTimer::getInstance()
{
  if ( s_timerInstance == 0 ) {
    s_timerInstance = new FpsTimer();
  }
  if ( s_timerInstance != 0 ) {
    numberOfInstances += 1;
  }
  return s_timerInstance;
}



/** Releases the timer obejct
 */
void FpsTimer::releaseInstance( FpsTimer* aTimer )
{
  if ( aTimer == 0 ) {
    return;
  }
  
  numberOfInstances -= 1;
  if ( numberOfInstances <= 0 ) {
    delete s_timerInstance;
    s_timerInstance = 0;
  }
}



///
/// Constructors, destructor and operators
/// ======================================

/** Constructor
 */
FpsTimer::FpsTimer()
{
}



/** Destructor
 */
FpsTimer::~FpsTimer()
{
}




///
/// Public methods
/// ==============

/** Resets the timer counter
 */
void FpsTimer::reset()
{
  updateCounter = 0;
}



/** Informs us that one update-round was done.
 */
void FpsTimer::updateDone()
{
  updateCounter -= 1;
  logicalUpdatesCountVar += 1;
}



///
/// Getter methods
/// ==============

/** Returns the number of updates you should do before next redraw.
 */
int FpsTimer::updatesToGo() const
{
  return updateCounter;
}



/** Returns the total number of updates we've done.
 */
unsigned int FpsTimer::totalUpdatesDone() const
{
  return numOfUpdates;
}



/** Returns the number of logical updates
 */
unsigned int FpsTimer::logicalUpdatesCount() const
{
  return logicalUpdatesCountVar;
}



///
/// Private or Protected methods
/// ============================

} // end of namespace
